# pico_i2c_lcd.py
# Implements a HD44780 character LCD connected via PCF8574 on I2C.

from lcd_api import LcdApi
from machine import I2C
import time

MASK_RS = 0x01
MASK_RW = 0x02
MASK_E = 0x04
SHIFT_BACKLIGHT = 3
MASK_BACKLIGHT = 1 << SHIFT_BACKLIGHT

class I2cLcd(LcdApi):
    def __init__(self, i2c, i2c_addr, num_lines, num_columns):
        self.i2c = i2c
        self.i2c_addr = i2c_addr
        self.i2c.writeto(self.i2c_addr, bytearray([0]))
        self.backlight = MASK_BACKLIGHT
        time.sleep_ms(20)

        self.hal_write_init_nibble(0x03)
        time.sleep_ms(5)
        self.hal_write_init_nibble(0x03)
        time.sleep_ms(1)
        self.hal_write_init_nibble(0x03)
        self.hal_write_init_nibble(0x02)

        super().__init__(num_lines, num_columns)

    def hal_write_init_nibble(self, nibble):
        byte = (nibble << 4) & 0xF0
        self.hal_write_byte(byte)

    def hal_backlight_on(self):
        self.backlight = MASK_BACKLIGHT
        self.i2c.writeto(self.i2c_addr, bytearray([self.backlight]))

    def hal_backlight_off(self):
        self.backlight = 0
        self.i2c.writeto(self.i2c_addr, bytearray([0]))

    def hal_write_command(self, cmd):
        self.hal_write(cmd, 0)

    def hal_write_data(self, data):
        self.hal_write(data, MASK_RS)

    def hal_write(self, data, mode):
        high_nibble = data & 0xF0
        low_nibble = (data << 4) & 0xF0
        self.hal_write_byte(high_nibble | mode)
        self.hal_write_byte(low_nibble | mode)

    def hal_write_byte(self, byte):
        self.i2c.writeto(self.i2c_addr, bytearray([byte | self.backlight | MASK_E]))
        self.i2c.writeto(self.i2c_addr, bytearray([(byte | self.backlight) & ~MASK_E]))
